// js/uiHandlers.js

/**
 * Konvertiert GDL in SVG basierend auf den aktuellen UI-Werten.
 * Greift auf globale Variablen zu: window.gdlInput, window.outWInput, window.outHInput, window.qualitySelect, window.parametersInput,
 *                                 convertGDLtoSVG (aus gdlConverter.js), window.currentSVG, window.currentErrors,
 *                                 window.previewBox, window.svgCodePreview, document.getElementById('converter-status').
 */
function handleConvert() {
	const width = parseInt(window.outWInput.value, 10) || 100; // Mit 'window.'-Präfix
	const height = parseInt(window.outHInput.value, 10) || 100; // Mit 'window.'-Präfix
	const quality = window.qualitySelect.value; // Mit 'window.'-Präfix
	const rawParams = window.parametersInput.value; // Mit 'window.'-Präfix
	
	// Die Funktion convertGDLtoSVG muss aus gdlConverter.js global verfügbar sein.
	const result = convertGDLtoSVG(window.gdlInput.value, width, height, quality, rawParams); // Mit 'window.'-Präfix
	
	// Globale Variablen aktualisieren
	window.currentSVG = result.svg;
	window.currentErrors = result.errors;
	
	// document.getElementById('converter-status') ist hier direkt und zum Zeitpunkt der Ausführung OK.
	const statusDiv = document.getElementById('converter-status');
	if (window.currentErrors.length > 0) {
		statusDiv.innerHTML = `<b>⚠️ GDL errors/notes::</b><br>${window.currentErrors.map(e => (e.length > 100 ? e.substring(0, 100) + '...' : e)).join("<br>")}`;
		statusDiv.style.display = 'block';
		console.warn("Conversion errors:", window.currentErrors);
	} else {
		statusDiv.style.display = 'none';
	}
	updatePreview();
}

/**
 * Aktualisiert die SVG-Vorschau und den SVG-Code.
 * Greift auf globale Variablen zu: window.currentSVG, window.previewBox, window.svgCodePreview, handleZoom, window.outWInput, window.outHInput.
 */
function updatePreview() {
	const svgBlob = new Blob([window.currentSVG], { type: 'image/svg+xml' });
	const url = URL.createObjectURL(svgBlob);
	const img = new Image();
	img.src = url;
	img.onload = () => { handleZoom(); URL.revokeObjectURL(url); };
	img.onerror = () => URL.revokeObjectURL(url);
	window.previewBox.innerHTML = ''; // Mit 'window.'-Präfix
	window.previewBox.appendChild(img); // Mit 'window.'-Präfix
	window.svgCodePreview.textContent = window.currentSVG; // Mit 'window.'-Präfix
}

/**
 * Wendet den Zoom auf die Vorschau an.
 * Greift auf globale Variablen zu: window.zoomSlider, window.zoomValue, window.previewBox, window.outWInput, window.outHInput.
 */
function handleZoom() {
	const zoom = parseFloat(window.zoomSlider.value); // Mit 'window.'-Präfix
	window.zoomValue.textContent = `${zoom.toFixed(1)}x`; // Mit 'window.'-Präfix
	const img = window.previewBox.querySelector('img'); // Mit 'window.'-Präfix
	if (img) {
		img.style.width = `${(parseInt(window.outWInput.value, 10) || 100) * zoom}px`; // Mit 'window.'-Präfix
		img.style.height = `${(parseInt(window.outHInput.value, 10) || 100) * zoom}px`; // Mit 'window.'-Präfix
	}
}

/**
 * Startet den Download der SVG-Datei.
 * Greift auf globale Variablen zu: window.currentSVG, window.filenameInput.
 */
function handleDownloadSVG() {
	if (!window.currentSVG) { alert("Bitte zuerst konvertieren."); return; }
	const a=document.createElement('a'), blob=new Blob([window.currentSVG],{type:"image/svg+xml"}), url=URL.createObjectURL(blob);
	a.href=url; a.download=`${window.filenameInput.value||'gdl_icon'}.svg`; a.click(); // Mit 'window.'-Präfix
	setTimeout(() => URL.revokeObjectURL(url), 100);
}

/**
 * Startet den Download der PNG-Datei.
 * Greift auf globale Variablen zu: window.currentSVG, window.outWInput, window.outHInput, window.qualitySelect, window.filenameInput.
 */
function handleDownloadPNG() {
	if (!window.currentSVG) { alert("Bitte zuerst konvertieren."); return; }
	const outW = parseInt(window.outWInput.value, 10) || 100; // Mit 'window.'-Präfix
	const outH = parseInt(window.outHInput.value, 10) || 100; // Mit 'window.'-Präfix
	const quality = window.qualitySelect.value; // Mit 'window.'-Präfix
	const scale = quality === 'highres' ? 4 : (quality === 'optimized' ? 2 : 1);
	const img = new Image(), svgBlob = new Blob([window.currentSVG], { type: 'image/svg+xml;charset=utf-8' }), url = URL.createObjectURL(svgBlob);
	img.onload = () => {
		const largeCanvas = document.createElement("canvas");
		largeCanvas.width = outW * scale; largeCanvas.height = outH * scale;
		largeCanvas.getContext("2d").drawImage(img, 0, 0, largeCanvas.width, largeCanvas.height);
		const finalCanvas = document.createElement("canvas");
		finalCanvas.width = outW; finalCanvas.height = outH;
		finalCanvas.getContext("2d").drawImage(largeCanvas, 0, 0, finalCanvas.width, finalCanvas.height);
		URL.revokeObjectURL(url);
		finalCanvas.toBlob(blob => {
			const a = document.createElement('a'), fileUrl = URL.createObjectURL(blob);
			a.href = fileUrl; a.download = `${window.filenameInput.value||'gdl_icon'}.png`; a.click(); // Mit 'window.'-Präfix
			setTimeout(() => URL.revokeObjectURL(fileUrl), 100);
		}, "image/png");
	};
	img.onerror = () => { URL.revokeObjectURL(url); alert("Error loading SVG for PNG export."); };
	img.src = url;
}

/**
 * Debounced-Version der handleConvert-Funktion, um häufige Aufrufe bei Texteingabe zu vermeiden.
 * Greift auf globale Variable zu: window.debounceTimer.
 */
function debouncedConvert() {
	clearTimeout(window.debounceTimer);
	window.debounceTimer = setTimeout(handleConvert, 300);
}
