// js/gdlConverter.js

/**
 * Expression evaluator for simple math (+, -, *, /) with variables.
 */
function evalExpr(expr, vars) {
	const tokens = [];
	const s = String(expr).replace(/\s+/g, '');
	let i = 0;
	while (i < s.length) {
		const c = s[i];
		if (/[0-9.]/.test(c)) { let j = i + 1; while (j < s.length && /[0-9.]/.test(s[j])) j++; tokens.push({ t: 'num', v: parseFloat(s.slice(i, j)) }); i = j; continue; }
		if (/[A-Za-z_]/.test(c)) { let j = i + 1; while (j < s.length && /[A-Za-z0-9_]/.test(s[j])) j++; const name = s.slice(i, j); if (!(name in vars)) throw `Variable not defined: ${name}`; tokens.push({ t: 'num', v: parseFloat(vars[name]) }); i = j; continue; }
		if ('+-*/()'.includes(c)) { tokens.push({ t: c }); i++; continue; }
		throw `Invalid character: "${c}"`;
	}
	const pTokens = [];
	for (let j = 0; j < tokens.length; j++) {
		const token = tokens[j], prev = j > 0 ? pTokens[pTokens.length - 1] : null;
		if (token.t === '-' && (!prev || ['(', '+', '-', '*', '/'].includes(prev.t))) {
			const next = tokens[j + 1]; if (next && next.t === 'num') { pTokens.push({ t: 'num', v: -next.v }); j++; } else { throw `Invalid unary minus`; }
		} else { pTokens.push(token); }
	}
	const out = [], ops = []; const prec = { '+': 1, '-': 1, '*': 2, '/': 2 };
	for (const tk of pTokens) {
		if (tk.t === 'num') { out.push(tk.v); }
		else if (tk.t in prec) { while (ops.length && (ops[ops.length - 1] in prec) && prec[ops[ops.length - 1]] >= prec[tk.t]) { const op = ops.pop(), b = out.pop(), a = out.pop(); out.push(op === '+' ? a + b : op === '-' ? a - b : op === '*' ? a * b : a / b); } ops.push(tk.t); }
		else if (tk.t === '(') { ops.push('('); }
		else if (tk.t === ')') { while (ops.length && ops[ops.length - 1] !== '(') { const op = ops.pop(), b = out.pop(), a = out.pop(); out.push(op === '+' ? a + b : op === '-' ? a - b : op === '*' ? a * b : a / b); } if (!ops.length) throw 'Mismatched parentheses'; ops.pop(); }
	}
	while (ops.length) { const op = ops.pop(), b = out.pop(), a = out.pop(); out.push(op === '+' ? a + b : op === '-' ? a - b : op === '*' ? a * b : a / b); }
	if (out.length !== 1) throw 'Expression unclear'; return out[0];
}

/**
 * Replaces variables in a single command line.
 */
function substituteVarsInCommand(line, vars) {
	const clean = line.trim(); if (!clean) return line;
	const firstSpace = clean.indexOf(" ");
	const cmd = (firstSpace === -1 ? clean : clean.slice(0, firstSpace)).toUpperCase();
	const argStr = (firstSpace === -1 ? "" : clean.slice(firstSpace + 1));
	if (!argStr.trim()) { return cmd; }

	const outArgs = argStr.split(",").map(tok => {
		tok = tok.trim();
		if (/^#([0-9a-f]{3}|[0-9a-f]{6})$/i.test(tok)) { return tok; }
		try {
			const v = evalExpr(tok, vars);
			return Number.isFinite(v) ? (Math.round(v * 1e6) / 1e6).toString() : tok;
		} catch {
			return tok;
		}
	});
	return cmd + " " + outArgs.join(", ");
}

/**
 * Preprocessor: Handles DEFINE, CALL, FOR loops, and variables.
 * HINWEIS: Akzeptiert jetzt initialVars.
 */
function preprocessScript(raw, initialVars, errsOut) {
	const lines = raw.split(/\r?\n/);
	const defs = {};
	const used = new Array(lines.length).fill(false);

	// 1. Collect all DEFINE blocks
	for (let i = 0; i < lines.length; i++) {
		const clean = lines[i].replace(/;+.*$/, '').trim();
		if (clean.toUpperCase().startsWith('DEFINE ')) {
			const name = clean.slice(7).trim();
			if (!/^[A-Za-z_][A-Za-z0-9_]*$/.test(name)) { errsOut.push(`Invalid DEFINE name: ${name}`); continue; }
			const body = [];
			let j = i + 1, found = false;
			for (; j < lines.length; j++) {
				if (lines[j].replace(/;+.*$/, '').trim().toUpperCase() === 'ENDDEFINE') { found = true; break; }
				body.push(lines[j]);
			}
			if (!found) { errsOut.push(`Missing ENDDEFINE for ${name}`); continue; }
			defs[name] = body;
			for (let k = i; k <= j; k++) used[k] = true;
			i = j;
		}
	}

	// 2. Recursively expand the script
	function expandBlock(blockLines, parentVars) {
		const out = [];
		const vars = { ...parentVars };

		for (let idx = 0; idx < blockLines.length; idx++) {
			const rawLine = blockLines[idx];
			const clean = rawLine.replace(/;+.*$/, '').trim();
			if (!clean) continue;
			const up = clean.toUpperCase();

			if (up.startsWith('FOR ')) {
				const match = clean.match(/^FOR\s+([A-Za-z_][A-Za-z0-9_]*)\s*=\s*(.+?)\s+TO\s+(.+?)\s+STEP\s+(.+?)\s*$/i);
				if (!match) { errsOut.push(`Invalid FOR syntax: ${clean}`); continue; }
				
				const [_, vName, aExpr, bExpr, sExpr] = match;
				let body = [], j = idx + 1, depth = 1, found = false;
				
				for (; j < blockLines.length; j++) {
					const lineJ = blockLines[j].replace(/;+.*$/, '').trim().toUpperCase();
					if (lineJ.startsWith('FOR ')) depth++;
					if (lineJ.startsWith('NEXT')) {
						const nextMatch = lineJ.match(/^NEXT(?:\s+([A-Za-z_][A-Za-z0-9_]*))?\s*$/i);
						if (nextMatch && --depth === 0) {
							if (nextMatch[1] && nextMatch[1].toLowerCase() !== vName.toLowerCase()) errsOut.push(`NEXT mismatch: Expected ${vName}`);
							found = true; break;
						}
					}
					body.push(blockLines[j]);
				}

				if (!found) { errsOut.push(`Missing NEXT for FOR ${vName}`); continue; }

				let start, end, step;
				try { start = evalExpr(aExpr, vars); end = evalExpr(bExpr, vars); step = evalExpr(sExpr, vars); }
				catch (e) { errsOut.push(`FOR expression error: ${e}`); idx = j; continue; }
				if (step === 0) { errsOut.push(`FOR step cannot be 0`); idx = j; continue; }

				if (step > 0) for (let v = start; v <= end + 1e-9; v += step) { vars[vName] = v; out.push(...expandBlock(body, vars)); }
				else for (let v = start; v >= end - 1e-9; v += step) { vars[vName] = v; out.push(...expandBlock(body, vars)); }
				
				delete vars[vName];
				idx = j;
				continue;
			}

			if (up.startsWith('NEXT')) { continue; }

			if (up.startsWith('CALL ')) {
				const name = clean.slice(5).trim();
				if (!defs[name]) { errsOut.push(`Unknown CALL: ${name}`); continue; }
				out.push(...expandBlock(defs[name], vars));
				continue;
			}

			const assignMatch = clean.match(/^([A-Za-z_][A-Za-z0-9_]*)\s*=\s*(.+)$/);
			if (assignMatch) {
				const [_, vName, expr] = assignMatch;
				try { vars[vName] = evalExpr(expr, vars); } catch (e) { errsOut.push(`Assignment failed: ${e}`); }
				continue;
			}

			out.push(substituteVarsInCommand(clean, vars));
		}
		return out;
	}
	
	const mainLines = lines.filter((_, i) => !used[i]);
	return { expanded: expandBlock(mainLines, initialVars) };
}

/**
 * Main converter function. Takes GDL script and returns SVG string and errors.
 * HINWEIS: Die Signatur wurde um rawParams erweitert.
 */
function convertGDLtoSVG(gdlScript, width, height, quality, rawParams) {
	const errors = [];
	const elements = [];
	const openGroups = [];
	
	// 1. Initial Vars Parsen
	const initialVars = {};
	if (rawParams) { 
		rawParams.split('\n').forEach(line => {
			const match = line.match(/^([A-Za-z_][A-Za-z0-9_]*)\s*=\s*(.+)$/);
			if (match) {
				try {
					// Verwende evalExpr, um Ausdrücke in Parametern zu unterstützen
					initialVars[match[1]] = evalExpr(match[2].trim(), {}); 
				} catch (e) {
					errors.push(`Parameter-Error: ${match[1]} = ${match[2]} (${e.message || e})`);
				}
			}
		});
	}

	// --- ZUSTAND FÜR TEXT (muss in der Funktion neu initialisiert werden) ---
	let currentFont = "Arial"; // GDL Default
	let currentFontStyle = 0;  // 0=normal, 1=bold, 2=italic, 3=bold+italic
	// -----------------------------------------------------------------------
	
	// 2. Preprocessing mit initialen Variablen (initialVars)
	const { expanded } = preprocessScript(gdlScript, initialVars, errors);

	const toNum = (v) => (typeof v === "string") ? parseFloat(v.replace(",", ".")) : v;
	const parseColor = (tok, fallback) => (typeof tok === "string" && /^#([0-9a-f]{3}|[0-9a-f]{6})$/i.test(tok.trim())) ? tok.trim() : fallback;
	const alignCoord = (coord, strokeWidth) => (quality === "standard" || strokeWidth % 2 !== 1) ? coord : Math.floor(coord) + 0.5;
	
	expanded.forEach((line, idx) => {
		const clean = line.trim(); if (!clean) return;
		const firstSpace = clean.indexOf(" "), cmd = (firstSpace === -1 ? clean : clean.slice(0, firstSpace)).toUpperCase();
		const args = (firstSpace === -1 ? "" : clean.slice(firstSpace + 1)).split(",").map(s => s.trim());
		const numAt = i => toNum(args[i]), colAt = (i, fb) => parseColor(args[i], fb);

		try {
				// Definierte GDL-Befehle für die Fehlererkennung
				const recognizedCommands = [
					"LINE2", "CIRCLE2", "RECT2", "POLY2", "ARROW", "ELLIPSE2", "ARC2",
					"STYLE", "TEXT2", "ADD2", "ROT2", "DEL"
				];
				
				if (cmd === "LINE2") {
					if (args.length < 5) throw "too few args";
					const d = numAt(4);
					const [x1, y1, x2, y2] = [alignCoord(numAt(0), d), alignCoord(numAt(1), d), alignCoord(numAt(2), d), alignCoord(numAt(3), d)];
					elements.push(`<line x1="${x1}" y1="${y1}" x2="${x2}" y2="${y2}" stroke="${colAt(5, "#000")}" stroke-width="${d}" />`);
				} else if (cmd === "CIRCLE2") {
					if (args.length < 4) throw "too few args";
					const d = numAt(3);
					const [x, y, r] = [alignCoord(numAt(0), d), alignCoord(numAt(1), d), numAt(2)];
					elements.push(`<circle cx="${x}" cy="${y}" r="${r}" stroke="${colAt(4, "#000")}" stroke-width="${d}" fill="${colAt(5, "none")}" />`);
				} else if (cmd === "RECT2") {
					if (args.length < 5) throw "too few args";
					const d = numAt(4);
					const [x1, y1, x2, y2] = [alignCoord(numAt(0), d), alignCoord(numAt(1), d), alignCoord(numAt(2), d), alignCoord(numAt(3), d)];
					const [x, y, w, h] = [Math.min(x1, x2), Math.min(y1, y2), Math.abs(x2 - x1), Math.abs(y2 - y1)];
					elements.push(`<rect x="${x}" y="${y}" width="${w}" height="${h}" stroke="${colAt(5, "#000")}" stroke-width="${d}" fill="${colAt(6, "none")}" />`);
				} else if (cmd === "POLY2") {
					if (args.length < 2) throw "invalid args";
					const n = parseInt(args[0]), needed=1+2*n;
					if (!n || n < 2 || args.length < needed + 1) throw "invalid n or coord count";
					const d = numAt(needed); const pts = [];
					for(let i=1; i<=2*n; i+=2) pts.push(`${alignCoord(numAt(i),d)},${alignCoord(numAt(i+1),d)}`);
					elements.push(`<polygon points="${pts.join(' ')}" stroke="${colAt(needed + 1, "#000")}" stroke-width="${d}" fill="${colAt(needed + 2, "none")}" />`);
				} else if (cmd === "ARROW") {
					if (args.length < 6) throw "too few args";
					const [tipX, tipY, rot, len, wid, d] = [numAt(0), numAt(1), numAt(2), numAt(3), numAt(4), numAt(5)];
					const r=rot*Math.PI/180, ux=Math.cos(r), uy=Math.sin(r), px=-uy, py=ux;
					const bcx=tipX-len*ux, bcy=tipY-len*uy, hw=wid/2;
					const p=`M ${tipX} ${tipY} L ${bcx+px*hw} ${bcy+py*hw} L ${bcx-px*hw} ${bcy-py*hw} Z`;
					elements.push(`<path d="${p}" stroke="${colAt(6,"#000")}" stroke-width="${d}" fill="${colAt(7,"none")}" />`);
				} else if (cmd === "ELLIPSE2") {
					if (args.length < 6) throw "too few args";
					const d = numAt(5);
					const [x, y, rx, ry, rot] = [alignCoord(numAt(0),d), alignCoord(numAt(1),d), numAt(2), numAt(3), numAt(4)];
					elements.push(`<ellipse cx="${x}" cy="${y}" rx="${rx}" ry="${ry}" transform="rotate(${rot} ${x} ${y})" stroke="${colAt(6,"#000")}" stroke-width="${d}" fill="${colAt(7,"none")}" />`);
				} else if (cmd === "ARC2") {
					if (args.length < 6) throw "too few args";
					const d = numAt(5);
					const [x, y, r, a1, a2] = [alignCoord(numAt(0),d), alignCoord(numAt(1),d), numAt(2), numAt(3), numAt(4)];
					const r1=a1*Math.PI/180, r2=a2*Math.PI/180;
					const p1_x=x+r*Math.cos(r1), p1_y=y+r*Math.sin(r1), p2_x=x+r*Math.cos(r2), p2_y=y+r*Math.sin(r2);
					const la=((a2-a1+360)%360)>180?1:0;
					elements.push(`<path d="M ${p1_x} ${p1_y} A ${r} ${r} 0 ${la} 1 ${p2_x} ${p2_y}" stroke="${colAt(6,"#000")}" stroke-width="${d}" />`);
				} else if (cmd === "STYLE") {
					if (args.length < 2) throw "too few args (expected STYLE 'Font', i)";
					const fontName = args[0].replace(/['"]/g, '').trim(); 
					const styleIndex = parseInt(args[1]) || 0;
					
					if(fontName) currentFont = fontName;
					currentFontStyle = styleIndex;
				} else if (cmd === "TEXT2") {
					if (args.length < 5) throw "too few args (expected TEXT2 x, y, h, i, 'Text', #Color)";
					const [x, y, h, alignIndex] = [numAt(0), numAt(1), numAt(2), numAt(3)];
					const text = args[4].replace(/['"]/g, '').trim(); 
					const color = colAt(5, "#000");
		
					const alignMap = { 1: 'start', 2: 'middle', 3: 'end' };
					const svgAnchor = alignMap[alignIndex] || 'start';
					
					const isBold = currentFontStyle === 1 || currentFontStyle === 3;
					const isItalic = currentFontStyle === 2 || currentFontStyle === 3;
					
					const flipTransform = `scale(1, -1) translate(0, ${-2 * y})`;
					
					const svgStyle = `font-family:'${currentFont.replace(/'/g, "\\'")}', sans-serif; ` +
										`font-size:${h}px; ` +
										`font-weight:${isBold ? 'bold' : 'normal'}; ` +
										`font-style:${isItalic ? 'italic' : 'normal'}; ` +
										`text-anchor:${svgAnchor}; ` +
										`fill:${color};`;
		
					elements.push(`<text x="${x}" y="${y}" style="${svgStyle}" transform="${flipTransform}">${text}</text>`);
				} else if (cmd === "ADD2") {
					if (args.length < 2) throw "too few args"; elements.push(`<g transform="translate(${numAt(0)} ${numAt(1)})">`); openGroups.push("g");
				} else if (cmd === "ROT2") {
					if (args.length < 1) throw "too few args"; elements.push(`<g transform="rotate(${-numAt(0)})">`); openGroups.push("g");
				} else if (cmd === "DEL") {
					const n = args.length ? parseInt(args[0] || "1") : 1;
					for (let i = 0; i < n; i++) { if (!openGroups.length) { errors.push(`Unmatched DEL on line ${idx + 1}`); break; } openGroups.pop(); elements.push(`</g>`); }
				} else if (recognizedCommands.includes(cmd)) {
					// Sollte hier nicht landen, da alle anerkannten Commands oben behandelt werden.
				} else if (cmd === "DEFINE" || cmd === "ENDDEFINE" || cmd === "CALL" || cmd === "FOR" || cmd === "NEXT") {
					// Preprocessor-Befehle, die im expanded-Script nicht auftauchen sollten.
				} else {
					// Unbekannter Befehl
					throw `Unknown command: ${cmd}`;
				}
			} catch (e) { 
				errors.push(`Line ${idx + 1} (${cmd}): ${e.message || e}`); 
			}
		});

	
	while(openGroups.length) { openGroups.pop(); elements.push(`</g>`); }

	const shapeRendering = quality !== "standard" ? ' shape-rendering="crispEdges"' : '';
	const svg = `<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 ${width} ${height}" width="${width}" height="${height}">
  <g transform="scale(1,-1) translate(0, -${height})" stroke-linecap="butt" stroke-linejoin="miter" fill="none"${shapeRendering}>
	${elements.join("\n    ")}
  </g>
</svg>`;
	return { svg, errors };
}
