// js/fileOperations.js

// Konstanten für localStorage
const GDL_STORAGE_KEY = 'gdl2svg_last_gdl_commands';
const PARAMS_STORAGE_KEY = 'gdl2svg_last_params';

/**
 * Speichert die aktuellen GDL-Befehle und Icon-Parameter in einer .gdlproject-Datei.
 * Verwendet File System Access API (wenn verfügbar) oder Fallback-Download.
 */
async function saveGdlProject() {
	// Greift auf globale Variablen zu: window.gdlInput, window.parametersInput, window.filenameInput
	const gdlCommands = window.gdlInput.value;
	const iconParameters = window.parametersInput.value;
	const filename = window.filenameInput.value || 'gdl_icon';

	const projectData = {
		gdlCommands: gdlCommands,
		iconParameters: iconParameters
	};
	const jsonString = JSON.stringify(projectData, null, 2);
	const blob = new Blob([jsonString], { type: 'application/json' });
	const suggestedName = `${filename}.gdlproject`;

	try {
		// --- Versuch 1: File System Access API ---
		if (window.showSaveFilePicker) {
			const fileHandle = await window.showSaveFilePicker({
				suggestedName: suggestedName,
				types: [{
					description: 'GDL Project File',
					accept: { 'application/json': ['.gdlproject'] },
				}],
			});

			const writableStream = await fileHandle.createWritable();
			await writableStream.write(blob);
			await writableStream.close();
			
			alert('GDL-Projekt erfolgreich gespeichert (File System Access API)!');
			return; // Erfolgreich gespeichert, Funktion beenden
		}
	} catch (error) {
		if (error.name === 'AbortError') {
			console.log('Speichervorgang vom Benutzer abgebrochen.');
			return; // Benutzer hat abgebrochen, Funktion beenden
		}
		console.warn('File System Access API Save failed:', error.message, 'Falling back to traditional download.');
		// Fehler, Fallback verwenden
	}

	// --- Fallback 2: Traditioneller Download (funktioniert überall) ---
	try {
		const url = URL.createObjectURL(blob);
		const a = document.createElement('a');
		a.href = url;
		a.download = suggestedName;
		document.body.appendChild(a); // Muss im DOM sein, um klickbar zu sein
		a.click();
		document.body.removeChild(a); // Nach dem Klick wieder entfernen
		URL.revokeObjectURL(url); // Freigabe des Object URL

		alert('GDL-Projekt erfolgreich gespeichert (traditioneller Download)!');
	} catch (error) {
		console.error('Fehler beim Speichern des GDL-Projekts (traditionell):', error);
		alert(`Fehler beim Speichern des GDL-Projekts: ${error.message}`);
	}
}


/**
 * Lädt GDL-Befehle und Icon-Parameter aus einer .gdlproject-Datei.
 * Verwendet File System Access API (wenn verfügbar) oder Fallback mit <input type="file">.
 */
async function loadGdlProject() {
	// Greift auf globale Variablen zu: window.gdlInput, window.parametersInput, window.filenameInput, window.debouncedConvert
	
	let file;
	try {
		// --- Versuch 1: File System Access API ---
		if (window.showOpenFilePicker) {
			const [fileHandle] = await window.showOpenFilePicker({
				types: [{
					description: 'GDL Project File',
					accept: { 'application/json': ['.gdlproject'] },
				}],
				multiple: false
			});
			file = await fileHandle.getFile();
		}
	} catch (error) {
		if (error.name === 'AbortError') {
			console.log('Ladevorgang vom Benutzer abgebrochen.');
			return;
		}
		console.warn('File System Access API Load failed:', error.message, 'Falling back to traditional file input.');
		// Fehler, Fallback verwenden
	}

	// --- Fallback 2: Traditioneller Datei-Upload mit <input type="file"> ---
	if (!file) {
		try {
			file = await new Promise((resolve, reject) => {
				const input = document.createElement('input');
				input.type = 'file';
				input.accept = '.gdlproject,application/json'; // Akzeptierte Dateitypen
				input.style.display = 'none'; // Verstecken des Inputs
				document.body.appendChild(input);

				input.addEventListener('change', (event) => {
					const selectedFile = event.target.files[0];
					if (selectedFile) {
						resolve(selectedFile);
					} else {
						reject(new Error('Keine Datei ausgewählt.'));
					}
					document.body.removeChild(input); // Input wieder entfernen
				});

				input.addEventListener('cancel', () => { // Nicht alle Browser feuern 'cancel'
					reject(new Error('Ladevorgang vom Benutzer abgebrochen.'));
					document.body.removeChild(input);
				});

				// Workaround für Browser, die kein 'cancel' Event feuern (z.B. Chrome, wenn Dialog geschlossen wird)
				// Beobachtet, ob der Dialog offen bleibt oder der Input seine Datei nicht bekommt
				setTimeout(() => {
					if (!input.files.length) {
						 // Kleine Verzögerung, um dem Benutzer Zeit zu geben, eine Datei auszuwählen
						 // Könnte noch verfeinert werden, um tatsächliches Abbrechen zu erkennen
						 // Wenn hier noch keine Datei ausgewählt ist, gehen wir davon aus, dass abgebrochen wurde
						// console.log("Timeout: no file selected, assuming user cancelled or closed dialog.");
						// reject(new Error('Ladevorgang vom Benutzer abgebrochen (Timeout).'));
						// Wir machen hier nichts, da es sonst den Promise rejected, obwohl vielleicht noch kein Klick auf "Abbrechen" war
						// Besser ist es, wenn der Benutzer wirklich auf "Abbrechen" klickt oder den Dialog schließt und der Promise offen bleibt.
					}
				}, 1000); // Eine Sekunde warten


				input.click(); // Öffnet den Dateiauswahldialog
			});
		} catch (error) {
			if (error.message === 'Ladevorgang vom Benutzer abgebrochen.') {
				 console.log(error.message);
				 return; // Benutzer hat abgebrochen
			}
			console.error('Fehler beim Öffnen des traditionellen Datei-Inputs:', error);
			alert(`Fehler beim Laden des GDL-Projekts: ${error.message}`);
			return;
		}
	}

	if (!file) {
		console.log("Keine Datei zum Laden verfügbar.");
		return; // Falls beide Methoden fehlschlagen oder der Benutzer abbricht
	}

	// Dateiinhalt lesen und UI aktualisieren
	try {
		const contents = await file.text();
		const projectData = JSON.parse(contents);

		if (projectData.gdlCommands !== undefined) {
			window.gdlInput.value = projectData.gdlCommands;
		}
		if (projectData.iconParameters !== undefined) {
			window.parametersInput.value = projectData.iconParameters;
		}
		
		window.filenameInput.value = file.name.replace(/\.gdlproject$/i, '');

		if (typeof window.debouncedConvert === 'function') {
			window.debouncedConvert();
		} else {
			console.warn("window.debouncedConvert ist nicht verfügbar. Manuelle Konvertierung erforderlich.");
		}
		alert('GDL-Projekt erfolgreich geladen!');

	} catch (error) {
		if (error.name === 'SyntaxError') {
			console.error('Fehler beim Parsen der GDL-Projektdatei (ungültiges JSON):', error);
			alert(`Fehler beim Laden: Die Datei ist kein gültiges GDL-Projekt-Format. ${error.message}`);
		} else {
			console.error('Fehler beim Laden des GDL-Projekts:', error);
			alert(`Fehler beim Laden des GDL-Projekts: ${error.message}`);
		}
	}
}


/**
 * Lädt GDL-Befehle und Icon-Parameter aus localStorage beim Start.
 * Greift auf globale Variablen zu: window.gdlInput, window.parametersInput
 */
function loadFromLocalStorage() {
	const storedGdl = localStorage.getItem(GDL_STORAGE_KEY);
	const storedParams = localStorage.getItem(PARAMS_STORAGE_KEY);

	if (storedGdl !== null) {
		window.gdlInput.value = storedGdl;
	}
	if (storedParams !== null) {
		window.parametersInput.value = storedParams;
	}
}

/**
 * Richtet Event-Listener für das Speichern in localStorage bei jeder Eingabe ein.
 * Greift auf globale Variablen zu: window.gdlInput, window.parametersInput
 */
function setupLocalStorageListeners() {
	window.gdlInput.addEventListener('input', () => {
		localStorage.setItem(GDL_STORAGE_KEY, window.gdlInput.value);
	});
	window.parametersInput.addEventListener('input', () => {
		localStorage.setItem(PARAMS_STORAGE_KEY, window.parametersInput.value);
	});
}
